<?php
/**
 * Plugin Name: Extensia Marketplace
 * Plugin URI: https://extensia.cloud/
 * Description: Displays organization listings and business detail pages using secure backend API calls.
 * Version: 1.4
 * Author: Satnam Singh Sandhu
 */

if (!defined('ABSPATH')) exit;

// Register plugin settings
function extensia_register_settings() {
    add_option('extensia_api_key_demo', '');
    add_option('extensia_api_key_live', '');
    add_option('extensia_api_url_demo', '');
    add_option('extensia_api_url_live', '');
    register_setting('extensia_options_group', 'extensia_api_key_demo');
    register_setting('extensia_options_group', 'extensia_api_key_live');
    register_setting('extensia_options_group', 'extensia_api_url_demo');
    register_setting('extensia_options_group', 'extensia_api_url_live');
}
add_action('admin_init', 'extensia_register_settings');

// Settings page
function extensia_register_options_page() {
    add_options_page('Extensia Marketplace Settings', 'Extensia Marketplace', 'manage_options', 'extensia-marketplace', 'extensia_options_page');
}
add_action('admin_menu', 'extensia_register_options_page');

function extensia_options_page() {
?>
<div class="wrap">
<h2>Extensia Marketplace Settings</h2>
<form method="post" action="options.php">
  <?php settings_fields('extensia_options_group'); ?>
  <table class="form-table">
    <tr><th><label for="extensia_api_key_demo">Demo API Key</label></th>
        <td><input type="text" name="extensia_api_key_demo" value="<?php echo get_option('extensia_api_key_demo'); ?>" size="50"/></td></tr>
    <tr><th><label for="extensia_api_url_demo">Demo API URL</label></th>
        <td><input type="text" name="extensia_api_url_demo" value="<?php echo get_option('extensia_api_url_demo'); ?>" size="50"/></td></tr>
    <tr><th><label for="extensia_api_key_live">Live API Key</label></th>
        <td><input type="text" name="extensia_api_key_live" value="<?php echo get_option('extensia_api_key_live'); ?>" size="50"/></td></tr>
    <tr><th><label for="extensia_api_url_live">Live API URL</label></th>
        <td><input type="text" name="extensia_api_url_live" value="<?php echo get_option('extensia_api_url_live'); ?>" size="50"/></td></tr>
  </table>
  <?php submit_button(); ?>
</form>
<h3>Usage:</h3>
<p>Use the following shortcodes:</p>
<code>[extensia_marketplace env="demo"]</code><br>
<code>[extensia_marketplace env="live"]</code>
</div>
<?php
}

// AJAX handler
add_action('wp_ajax_extensia_fetch_orgs', 'extensia_fetch_orgs');
add_action('wp_ajax_nopriv_extensia_fetch_orgs', 'extensia_fetch_orgs');
function extensia_fetch_orgs() {
    $env = sanitize_text_field($_GET['env'] ?? 'live');
    $key = $env === 'demo' ? get_option('extensia_api_key_demo') : get_option('extensia_api_key_live');
    $url = $env === 'demo' ? get_option('extensia_api_url_demo') : get_option('extensia_api_url_live');

    $response = wp_remote_get($url, [
        'headers' => [
            'Content-Type' => 'application/json',
            'key' => $key
        ]
    ]);

    if (is_wp_error($response)) {
        wp_send_json_error(['message' => 'Failed to contact API.']);
    }

    $body = wp_remote_retrieve_body($response);
    wp_send_json_success(json_decode($body, true));
}

// Listing shortcode
function extensia_org_listing_shortcode($atts) {
    $atts = shortcode_atts(['env' => 'live'], $atts);
    $env = esc_js($atts['env']);
    $ajax_url = admin_url('admin-ajax.php');

    ob_start(); ?>
    <style>
    .ext-search-box { max-width: 400px; margin: 20px auto; text-align: center; }
    .ext-search-box input { width: 100%; padding: 10px; font-size: 16px; border-radius: 8px; }
    .ext-org-listing { display: grid; grid-template-columns: repeat(auto-fill, minmax(300px, 1fr)); gap: 20px; margin-top: 20px; }
    .ext-org-card { position: relative; background: #fff; border-radius: 12px; box-shadow: 0 2px 6px rgba(0,0,0,0.1); padding: 16px; transition: transform 0.2s ease; cursor: pointer; }
    .ext-org-card:hover { transform: scale(1.02); }
    .ext-org-image { width: 100%; height: 180px; object-fit: cover; border-radius: 8px; margin-bottom: 12px; }
    .ext-org-name { font-size: 20px; font-weight: bold; margin-bottom: 8px; color: #333; }
    .ext-org-info { font-size: 14px; color: #555; margin-bottom: 4px; }
    .ext-org-contact { margin-top: 10px; font-size: 14px; color: #4f6275; }
    .ext-tag-overlay { position: absolute; top: 16px; right: 16px; background-color: #007bff; color: white; padding: 4px 10px; border-radius: 12px; font-size: 12px; font-weight: bold; }
    @media screen and (max-width: 600px) {
        .ext-org-listing { grid-template-columns: 1fr; }
    }
    </style>
    <div class="ext-search-box"><input type="text" id="orgSearchInput" placeholder="Search businesses..."></div>
    <div class="ext-org-listing" id="orgListContainer"><p>Loading organizations...</p></div>
    <script>
    let allOrgs = [];
    async function fetchExtensiaOrgs() {
        const res = await fetch("<?php echo $ajax_url; ?>?action=extensia_fetch_orgs&env=<?php echo $env; ?>");
        const json = await res.json();
        let orgs = typeof json.data === 'string' ? JSON.parse(json.data) : json.data;
        allOrgs = Array.isArray(orgs) ? orgs : [];
        renderOrgs(allOrgs);
    }

    function renderOrgs(orgs) {
        const container = document.getElementById('orgListContainer');
        container.innerHTML = '';
        if (!orgs.length) {
            container.innerHTML = '<p>No organizations found.</p>';
            return;
        }
        orgs.forEach(org => {
            const imageUrl = org.BackgroundImage || 'https://picsum.photos/400/200?grayscale';
            const tier = org.Tier || '';
            const showTag = ["Individual", "Business", "Enterprise"].includes(tier);
            const tag = showTag ? '<div class="ext-tag-overlay">Member</div>' : '';
            const card = `<div class='ext-org-card' onclick="location.href='?extensia_detail=1&org_id=${org.OrganizationId}&biz_id=${org.BusinessId}&env=<?php echo $env; ?>'">
                ${tag}
                <img src="${imageUrl}" alt="Org Logo" class="ext-org-image" />
                <div class="ext-org-name">${org.Name}</div>
                <div class="ext-org-info"><strong>Website:</strong> <a href="${org.Website}" target="_blank">${org.Website}</a></div>
                <div class="ext-org-info"><strong>Keywords:</strong> ${org.SearchKeywords}</div>
                <div class="ext-org-contact">Phone: ${org.PhoneNumber}</div>
            </div>`;
            container.insertAdjacentHTML('beforeend', card);
        });
    }

    document.getElementById('orgSearchInput').addEventListener('input', e => {
        const q = e.target.value.toLowerCase();
        const filtered = allOrgs.filter(o => o.Name?.toLowerCase().includes(q) || o.SearchKeywords?.toLowerCase().includes(q));
        renderOrgs(filtered);
    });

    fetchExtensiaOrgs();
    </script>
<?php
    return ob_get_clean();
}
add_shortcode('extensia_marketplace', 'extensia_org_listing_shortcode');

// Business detail template routing
add_filter('template_include', function($template) {
    if (isset($_GET['extensia_detail']) && $_GET['extensia_detail'] === '1') {
        return plugin_dir_path(__FILE__) . 'templates/business-detail-template.php';
    }
    return $template;
});
